/** @file   glyphset.cpp
 * @brief   Implementation of GlyphSet - class.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */

#include "glyphset.h"

using std::istream;
using std::string;
using std::map;

namespace eng2d {


///
/// Static members, constants and datatypes
/// =======================================




///
/// Constructors, destructor and operators
/// ======================================
/** Constructor
 */
GlyphSet::GlyphSet( BITMAP* pB, istream& rIn ) :
  m_gridX( 32 ),
  m_gridY( 48 ),
  m_glyphHeight( 32 ),
  m_id(""),
  m_bad( true )
{
  int ret = this->readGlyphData( pB, rIn );
}



/** Destructor
 */
GlyphSet::~GlyphSet()
{
  // Destroy the bitmaps created for the glyphs.
  map<int, Glyph*>::const_iterator it = m_characters.begin();
  while ( it != m_characters.end() ) {
    if ( it->second != 0 ) {
      delete it->second;
    }
  }
  m_characters.clear();
}




///
/// Public methods
/// ==============




///
/// Getter methods
/// ==============

/** Tells does this glyphset has the given character
 */
bool GlyphSet::hasGlyph( int c ) const
{
  if ( this->getGlyphData( c ) != 0 ) {
    return true;
  }
  return false;
}



/** Returns the width of the requested character
 */
int GlyphSet::glyphWidth( int c ) const
{
  Glyph* pG = this->getGlyphData( c );
  if ( pG == 0 ) {
    return 0;
  }
  return pG->width();
}



/** Returns the bitmap containing the graphics
 */
BITMAP* GlyphSet::gfx( int c ) const
{
  Glyph* pG = this->getGlyphData( c );
  if ( pG == 0 ) {
    return 0;
  }
  return pG->gfx();
}



/** Tells if glyphset is ok to be used.
 */
bool GlyphSet::bad() const
{
  return false;
}



/** Returns the size of the glyphs
 */
int GlyphSet::glyphSize() const
{
  return m_glyphHeight;
}



/** Returns the id of this glyphset
 */
string GlyphSet::id() const
{
  return m_id;
}




///
/// Private or Protected methods
/// ============================

/** Reads the glyph data
 */
int GlyphSet::readGlyphData( BITMAP* pB, istream& rIn )
{
  if ( pB == 0 ) {
    alert("null-bitmap",0,0, "ok",0, 0,0);
  }
  
  m_gridX = 0;
  m_gridY = 0;
  while ( true ) {
    if ( rIn.eof() == true ) {
      return -1;
    }
    
    string tmp;
    rIn >> tmp;
    if ( tmp == "height:" ) {
      rIn >> m_glyphHeight;
      
    } else if ( tmp == "id:" ) {
      rIn >> m_id;
      
    } else if ( tmp == "grid:" ) {
      rIn >> m_gridX;
      rIn >> m_gridY;
      
    } else if ( tmp == "<glyphset>" ) {
      int ret = this->readGlyphSet( rIn );
      
    } else if ( tmp == "</font>" ) {
      break;
      
    } else if ( tmp == "#" ) {
      rIn.ignore( 4096, '\n' );
      
    }
  }

  this->createCharacters( pB );
  return 0;
}



/** Reads the glyphset
 */
int GlyphSet::readGlyphSet( istream& rIn )
{
  int index = 0;
  while ( true ) {
    if ( rIn.eof() == true ) {
      return -1;
    }
    
    string tmp;
    rIn >> tmp;
    if ( tmp == "</glyphset>" ) {
      break;
      
    } else {
      char c;
      if ( tmp == "SPACE" ) {
        c = ' ';
      } else {
        c = tmp.at(0);
      }
      
      rIn >> tmp;
      if ( tmp != ":" ) {
        // Invalid lineformat
        return -1;
      }
      rIn >> tmp;
      // Create the glyph and add it to the glyph map.
      Glyph* pG = new Glyph( index, static_cast<int>(c), atoi(tmp.c_str()) );
      
      index += 1;
      m_characters[ static_cast<int>( c ) ] = pG;
    }
  }
  return 0;
}



/** Create the characters
 */
void GlyphSet::createCharacters( BITMAP* pB )
{
  map<int, Glyph*>::iterator it = m_characters.begin();
  int counter = 0;
  while ( it != m_characters.end() ) {
    BITMAP* pGfx = this->grabCharacter( pB, it->second->m_index );
    it->second->setGfx( pGfx );
    
    it++;
    counter += 1;
    if ( counter > 400 ) {
      break;
    }
  }
}



/** Grabs the character
 */
BITMAP* GlyphSet::grabCharacter( BITMAP* pB, int index )
{
  int x = index * m_gridX;
  
  BITMAP* pCharB = create_bitmap( m_gridX, m_gridY );
  if ( pCharB != 0 ) {
    clear_to_color( pCharB, makecol(255,0,255) );
    blit( pB, pCharB, x,0, 0,0, m_gridX, m_gridY );
  }
  return pCharB;
}



/** Returns the data of requested glyph
 */
Glyph* GlyphSet::getGlyphData( int c ) const
{
  map<int, Glyph*>::const_iterator it = m_characters.find( c );
  if ( it == m_characters.end() ) {
    return 0;
  }
  return it->second;
}


} // end of namespace
